/**
 * @file: svg_font.h
 *
 * @version: $Id: svg_font.h,v 1.28.6.1 2011/10/24 15:07:04 efriedrich Exp $
 *
 * This header file declares the functions and types that are provided
 * by the SVGFont
 * <detailed description>
 * @component: SVGFont
 *
 * @author      D.Langner  / ADIT-G / SWG / dlangner@de.adit-jv.com
 *              C.Spetzler / ADIT-G / SWG / cpspetzler@de.adit-jv.com
 *
 * @copyright: (c) 2003 - 2006 ADIT Corporation
 *
 * @see <related items>
 *
 ***********************************************************************/
#ifndef SVG_FONT_H
#define SVG_FONT_H


#ifdef __cplusplus
extern "C" {
#endif

#include    "svg_typedef.h"
#include    "svg_error.h"                     /* Error constant value define */
#include    "EGL/egl.h"


/**
 *  Constant definition for Font Manager
 */

/**
 * @def     Font attributes bit pattern.
 */
#define SVG_FONT_FIXED                      0x00000001     /**< Fixed font width          */
#define SVG_FONT_PROP                       0x00000002     /**< Proportional font X       */
#define SVG_FONT_SCALABLE                   0x00000004     /**< Scalable font             */
#define SVG_FONT_PLAIN                      0x00000008     /**< Standard font             */
#define SVG_FONT_BOLD                       0x00000010     /**< Bold font                 */
#define SVG_FONT_ITALIC                     0x00000020     /**< Italic font               */

/**
 * @def     Rendering Settings Parameter.
 */
#define SVG_FONT_HINTING                    0x00000040     /**< ON: Hinting enable        */
#define SVG_FONT_ANTIALIAS                  0x00000080     /**< ON: Antialias enable      */
#define SVG_FONT_NO_ENCLOSING_SPACES        0x00000100     /**< ON: omit leading and trailing spaces for
                                                                    font drawing and bounding box calculation */
#define SVG_FONT_OPACITY                    0x00000200     /**< ON: Blend fonts with given alpha  */
#define SVG_FONT_OUTLINE_SPACE_EXTENSION    0x00000400     /**< ON: letter space will be extended
                                                                    when font drawing with outline / 2color */
#define SVG_FONT_USE_REPLACEMENT_CHAR       0x00000800     /**< ON: use the char set by svgFontSetReplacementChar
                                                                    when an unknown char is found in string */
#define SVG_FONT_USE_CURRENT_MATRIX         0x00001000     /**< ON: use the transformation matrix from upper API */
#define SVG_FONT_VERTICAL                   0x00002000     /**< ON: Vertical OFF: Horiz.  */
#define SVG_FONT_KERNING                    0x00004000     /**< ON: Kerning enable        */
#define SVG_FONT_SCALE                      0x00008000     /**< ON: Scaling enable        */
#define SVG_FONT_USE_CURRENT_PROGRAM        0x00010000     /**< ON GLES2 shader program of application should be used */
#define SVG_FONT_2COLOR                     0x00020000     /**< ON: Fast Outlining enable */
#define SVG_FONT_OUTLINE                    0x00040000     /**< ON: FT Outlining enable   */
#define SVG_FONT_MIPMAPPING                 0x00080000     /**< ON: Mipmapping enabled    */
#define SVG_FONT_INVALID_SETTING			0x000dFFFF

/**
 * @def     Supported flash fonts.
 *
 * Declare fixed ids for faster access in user applications.
 */
#define SVG_PLUGIN_FREETYPE     1   /**< Fixed id for module Freetype.        */

/**
 * @def     Set pointer defines for some getter functions.
 *
 * For functions: svgGetAvailableFontModules and svgGetAvailableFonts.
 */
#define SVG_TOP                 1   /**< Set pointer to top position.         */
#define SVG_NEXT                2   /**< Set pointer to next position.        */

#define SVG_OPAQUE_FONT         0xFF /*opaque value*/

/**
 * @struct grl_flash_font
 *
 * This struct is used to notify global flash fonts for user font modules.\n
 *
 * @param SVGInt8*      : Declare the suffix.
 * @param SVGUint8*     : Start address of font.
 * @param SVGInt32      : Size of font.
 *
 * @see   svg_font_config.h
 */
typedef struct flash_font_struct{
    SVGInt8*        suffix;
    const SVGUint8* start_address;
    SVGUint32       length;
} grl_flash_font;


/* SVGRenderDirection*/

typedef enum {
  SVG_RENDER_DIRECTION_INVALID = 0,
  SVG_RENDER_DIRECTION_LTR = 4,
  SVG_RENDER_DIRECTION_RTL,
  SVG_RENDER_DIRECTION_TTB,
  SVG_RENDER_DIRECTION_BTT
} SVGRenderDirection;

/* SVGLangScript */
#define SVG_TAG(c1,c2,c3,c4) ((SVGUint32)((((SVGUint8)(c1))<<24)|(((SVGUint8)(c2))<<16)|(((SVGUint8)(c3))<<8)|((SVGUint8)(c4))))
/**
 *  @enum Enumeration definition for harfbuzz library.
 */

typedef enum
{
    SVG_TAG_NONE                                       =  0,
  SVG_SCRIPT_COMMON				= SVG_TAG ('Z','y','y','y'),
  SVG_SCRIPT_INHERITED			= SVG_TAG ('Z','i','n','h'),
  SVG_SCRIPT_UNKNOWN			= SVG_TAG ('Z','z','z','z'),

  SVG_SCRIPT_ARABIC				= SVG_TAG ('A','r','a','b'),
  SVG_SCRIPT_ARMENIAN			= SVG_TAG ('A','r','m','n'),
  SVG_SCRIPT_BENGALI			= SVG_TAG ('B','e','n','g'),
  SVG_SCRIPT_CYRILLIC			= SVG_TAG ('C','y','r','l'),
  SVG_SCRIPT_DEVANAGARI			= SVG_TAG ('D','e','v','a'),
  SVG_SCRIPT_GEORGIAN			= SVG_TAG ('G','e','o','r'),
  SVG_SCRIPT_GREEK				= SVG_TAG ('G','r','e','k'),
  SVG_SCRIPT_GUJARATI			= SVG_TAG ('G','u','j','r'),
  SVG_SCRIPT_GURMUKHI			= SVG_TAG ('G','u','r','u'),
  SVG_SCRIPT_HANGUL				= SVG_TAG ('H','a','n','g'),
  SVG_SCRIPT_HAN				= SVG_TAG ('H','a','n','i'),
  SVG_SCRIPT_HEBREW				= SVG_TAG ('H','e','b','r'),
  SVG_SCRIPT_HIRAGANA			= SVG_TAG ('H','i','r','a'),
  SVG_SCRIPT_KANNADA			= SVG_TAG ('K','n','d','a'),
  SVG_SCRIPT_KATAKANA			= SVG_TAG ('K','a','n','a'),
  SVG_SCRIPT_LAO				= SVG_TAG ('L','a','o','o'),
  SVG_SCRIPT_LATIN				= SVG_TAG ('L','a','t','n'),
  SVG_SCRIPT_MALAYALAM			= SVG_TAG ('M','l','y','m'),
  SVG_SCRIPT_ORIYA				= SVG_TAG ('O','r','y','a'),
  SVG_SCRIPT_TAMIL				= SVG_TAG ('T','a','m','l'),
  SVG_SCRIPT_TELUGU				= SVG_TAG ('T','e','l','u'),
  SVG_SCRIPT_THAI				= SVG_TAG ('T','h','a','i'),

  SVG_SCRIPT_TIBETAN			= SVG_TAG ('T','i','b','t'),

  SVG_SCRIPT_BOPOMOFO			= SVG_TAG ('B','o','p','o'),
  SVG_SCRIPT_BRAILLE			= SVG_TAG ('B','r','a','i'),
  SVG_SCRIPT_CANADIAN_SYLLABICS		= SVG_TAG ('C','a','n','s'),
  SVG_SCRIPT_CHEROKEE			= SVG_TAG ('C','h','e','r'),
  SVG_SCRIPT_ETHIOPIC			= SVG_TAG ('E','t','h','i'),
  SVG_SCRIPT_KHMER				= SVG_TAG ('K','h','m','r'),
  SVG_SCRIPT_MONGOLIAN			= SVG_TAG ('M','o','n','g'),
  SVG_SCRIPT_MYANMAR			= SVG_TAG ('M','y','m','r'),
  SVG_SCRIPT_OGHAM				= SVG_TAG ('O','g','a','m'),
  SVG_SCRIPT_RUNIC				= SVG_TAG ('R','u','n','r'),
  SVG_SCRIPT_SINHALA			= SVG_TAG ('S','i','n','h'),
  SVG_SCRIPT_SYRIAC				= SVG_TAG ('S','y','r','c'),
  SVG_SCRIPT_THAANA				= SVG_TAG ('T','h','a','a'),
  SVG_SCRIPT_YI					= SVG_TAG ('Y','i','i','i'),

  SVG_SCRIPT_DESERET			= SVG_TAG ('D','s','r','t'),
  SVG_SCRIPT_GOTHIC				= SVG_TAG ('G','o','t','h'),
  SVG_SCRIPT_OLD_ITALIC			= SVG_TAG ('I','t','a','l'),

  SVG_SCRIPT_BUHID				= SVG_TAG ('B','u','h','d'),
  SVG_SCRIPT_HANUNOO			= SVG_TAG ('H','a','n','o'),
  SVG_SCRIPT_TAGALOG			= SVG_TAG ('T','g','l','g'),
  SVG_SCRIPT_TAGBANWA			= SVG_TAG ('T','a','g','b'),

  SVG_SCRIPT_CYPRIOT			= SVG_TAG ('C','p','r','t'),
  SVG_SCRIPT_LIMBU				= SVG_TAG ('L','i','m','b'),
  SVG_SCRIPT_LINEAR_B			= SVG_TAG ('L','i','n','b'),
  SVG_SCRIPT_OSMANYA			= SVG_TAG ('O','s','m','a'),
  SVG_SCRIPT_SHAVIAN			= SVG_TAG ('S','h','a','w'),
  SVG_SCRIPT_TAI_LE				= SVG_TAG ('T','a','l','e'),
  SVG_SCRIPT_UGARITIC			= SVG_TAG ('U','g','a','r'),

  SVG_SCRIPT_BUGINESE			= SVG_TAG ('B','u','g','i'),
  SVG_SCRIPT_COPTIC				= SVG_TAG ('C','o','p','t'),
  SVG_SCRIPT_GLAGOLITIC			= SVG_TAG ('G','l','a','g'),
  SVG_SCRIPT_KHAROSHTHI			= SVG_TAG ('K','h','a','r'),
  SVG_SCRIPT_NEW_TAI_LUE		= SVG_TAG ('T','a','l','u'),
  SVG_SCRIPT_OLD_PERSIAN		= SVG_TAG ('X','p','e','o'),
  SVG_SCRIPT_SYLOTI_NAGRI		= SVG_TAG ('S','y','l','o'),
  SVG_SCRIPT_TIFINAGH			= SVG_TAG ('T','f','n','g'),

  SVG_SCRIPT_BALINESE			= SVG_TAG ('B','a','l','i'),
  SVG_SCRIPT_CUNEIFORM			= SVG_TAG ('X','s','u','x'),
  SVG_SCRIPT_NKO				= SVG_TAG ('N','k','o','o'),
  SVG_SCRIPT_PHAGS_PA			= SVG_TAG ('P','h','a','g'),
  SVG_SCRIPT_PHOENICIAN			= SVG_TAG ('P','h','n','x'),

  SVG_SCRIPT_CARIAN				= SVG_TAG ('C','a','r','i'),
  SVG_SCRIPT_CHAM				= SVG_TAG ('C','h','a','m'),
  SVG_SCRIPT_KAYAH_LI			= SVG_TAG ('K','a','l','i'),
  SVG_SCRIPT_LEPCHA				= SVG_TAG ('L','e','p','c'),
  SVG_SCRIPT_LYCIAN				= SVG_TAG ('L','y','c','i'),
  SVG_SCRIPT_LYDIAN				= SVG_TAG ('L','y','d','i'),
  SVG_SCRIPT_OL_CHIKI			= SVG_TAG ('O','l','c','k'),
  SVG_SCRIPT_REJANG				= SVG_TAG ('R','j','n','g'),
  SVG_SCRIPT_SAURASHTRA			= SVG_TAG ('S','a','u','r'),
  SVG_SCRIPT_SUNDANESE			= SVG_TAG ('S','u','n','d'),
  SVG_SCRIPT_VAI				= SVG_TAG ('V','a','i','i'),

  SVG_SCRIPT_AVESTAN			= SVG_TAG ('A','v','s','t'),
  SVG_SCRIPT_BAMUM				= SVG_TAG ('B','a','m','u'),
  SVG_SCRIPT_EGYPTIAN_HIEROGLYPHS	= SVG_TAG ('E','g','y','p'),
  SVG_SCRIPT_IMPERIAL_ARAMAIC		= SVG_TAG ('A','r','m','i'),
  SVG_SCRIPT_INSCRIPTIONAL_PAHLAVI	= SVG_TAG ('P','h','l','i'),
  SVG_SCRIPT_INSCRIPTIONAL_PARTHIAN	= SVG_TAG ('P','r','t','i'),
  SVG_SCRIPT_JAVANESE			= SVG_TAG ('J','a','v','a'),
  SVG_SCRIPT_KAITHI			= SVG_TAG ('K','t','h','i'),
  SVG_SCRIPT_LISU			= SVG_TAG ('L','i','s','u'),
  SVG_SCRIPT_MEETEI_MAYEK		= SVG_TAG ('M','t','e','i'),
  SVG_SCRIPT_OLD_SOUTH_ARABIAN		= SVG_TAG ('S','a','r','b'),
  SVG_SCRIPT_OLD_TURKIC			= SVG_TAG ('O','r','k','h'),
  SVG_SCRIPT_SAMARITAN			= SVG_TAG ('S','a','m','r'),
  SVG_SCRIPT_TAI_THAM			= SVG_TAG ('L','a','n','a'),
  SVG_SCRIPT_TAI_VIET			= SVG_TAG ('T','a','v','t'),

  SVG_SCRIPT_BATAK			= SVG_TAG ('B','a','t','k'),
  SVG_SCRIPT_BRAHMI			= SVG_TAG ('B','r','a','h'),
  SVG_SCRIPT_MANDAIC			= SVG_TAG ('M','a','n','d'),

  SVG_SCRIPT_CHAKMA			= SVG_TAG ('C','a','k','m'),
  SVG_SCRIPT_MEROITIC_CURSIVE		= SVG_TAG ('M','e','r','c'),
  SVG_SCRIPT_MEROITIC_HIEROGLYPHS	= SVG_TAG ('M','e','r','o'),
  SVG_SCRIPT_MIAO			= SVG_TAG ('P','l','r','d'),
  SVG_SCRIPT_SHARADA			= SVG_TAG ('S','h','r','d'),
  SVG_SCRIPT_SORA_SOMPENG		= SVG_TAG ('S','o','r','a'),
  SVG_SCRIPT_TAKRI			= SVG_TAG ('T','a','k','r'),

  SVG_SCRIPT_BASSA_VAH			= SVG_TAG ('B','a','s','s'),
  SVG_SCRIPT_CAUCASIAN_ALBANIAN		= SVG_TAG ('A','g','h','b'),
  SVG_SCRIPT_DUPLOYAN			= SVG_TAG ('D','u','p','l'),
  SVG_SCRIPT_ELBASAN			= SVG_TAG ('E','l','b','a'),
  SVG_SCRIPT_GRANTHA			= SVG_TAG ('G','r','a','n'),
  SVG_SCRIPT_KHOJKI			= SVG_TAG ('K','h','o','j'),
  SVG_SCRIPT_KHUDAWADI			= SVG_TAG ('S','i','n','d'),
  SVG_SCRIPT_LINEAR_A			= SVG_TAG ('L','i','n','a'),
  SVG_SCRIPT_MAHAJANI			= SVG_TAG ('M','a','h','j'),
  SVG_SCRIPT_MANICHAEAN			= SVG_TAG ('M','a','n','i'),
  SVG_SCRIPT_MENDE_KIKAKUI		= SVG_TAG ('M','e','n','d'),
  SVG_SCRIPT_MODI			= SVG_TAG ('M','o','d','i'),
  SVG_SCRIPT_MRO				= SVG_TAG ('M','r','o','o'),
  SVG_SCRIPT_NABATAEAN			= SVG_TAG ('N','b','a','t'),
  SVG_SCRIPT_OLD_NORTH_ARABIAN		= SVG_TAG ('N','a','r','b'),
  SVG_SCRIPT_OLD_PERMIC			= SVG_TAG ('P','e','r','m'),
  SVG_SCRIPT_PAHAWH_HMONG		= SVG_TAG ('H','m','n','g'),
  SVG_SCRIPT_PALMYRENE			= SVG_TAG ('P','a','l','m'),
  SVG_SCRIPT_PAU_CIN_HAU			= SVG_TAG ('P','a','u','c'),
  SVG_SCRIPT_PSALTER_PAHLAVI		= SVG_TAG ('P','h','l','p'),
  SVG_SCRIPT_SIDDHAM			= SVG_TAG ('S','i','d','d'),
  SVG_SCRIPT_TIRHUTA			= SVG_TAG ('T','i','r','h'),
  SVG_SCRIPT_WARANG_CITI			= SVG_TAG ('W','a','r','a'),

  /* No script set. */
  SVG_SCRIPT_INVALID				= SVG_TAG_NONE,
} SVGLangScript;

/**
 *  @enum Enumeration definition for Font Manager.
 */
typedef enum {
    SVG_ENCODING_MIN,                          /* for Parameter check         */
    SVG_ENCODING_ASCII,                       /* ASCII (8bit)                */
    SVG_ENCODING_UNICODE,                     /* UNICODE                     */
    SVG_ENCODING_UTF8,                        /* UTF-8                       */
    SVG_ENCODING_SJIS,                        /* SJIS (Japanese)             */
    SVG_ENCODING_MAX                          /* for Parameter check         */
} SVGEncoding;

/**
 *  @enum Enumeration definition for Origin Type.
 */
typedef enum {
    SVG_ORIGIN_MIN,                            /* for Parameter check         */
    SVG_ORIGIN_BASELINE,
    SVG_ORIGIN_TOP,
    SVG_ORIGIN_BOTTOM,
    SVG_ORIGIN_MAX                            /* for Parameter check         */
} SVGOriginType;


/**
 *  @struct Structure definition for Font Manager.
 */
typedef struct {
    SVGUint8                fontName[64];     /* Font name                   */
    SVGUint8                filename[256];    /* File name                   */
    SVGUint8                allowedSizes[32];
    SVGUint8                charMaps[16];
    SVGUint32               attributes;       /* Font attribute              */
    SVGUint32               lengthSizes;
    SVGUint32               lengthMaps;
    SVGUint32               faceType;
	SVGUint32				faceIndex;
} SVGFont;


/**
 *  @struct Structure definition of SVGFontModule information.
 */
typedef struct{
    SVGUint32                   id;
    SVGUint8                    name[64];
                     /* Pointer to the function pointer table of font plugin */
    struct _grl_fontmodule_fp   *p_grl_fp_table;
} SVGFontModule;


/**
 *  @struct Structure definition of internal module information.
 */
typedef struct  {
    SVGFontModule               *p_module;
    SVGFont                     *p_svgFlashFonts;
    SVGUint32                   svgFlashFontsMax;
    SVGUint32                   flashFontCnt;
    SVGFont                     *p_svgFileFonts;
    SVGUint32                   svgFileFontsMax;
    SVGUint32                   fileFontCnt;
} grl_module_info;


/**
 *  @struct This struct is used for internal module information handling.
 */
typedef struct {
    void *                          next;
    grl_module_info             *p_module_info;
} grl_module_info_list;

/**
 *  @struct
 *
 *  This struct is used for internal module information handling, too.
 *  But the dependency lies on the context (for multitasking).
 */
typedef struct {
    void *                          next;
    grl_module_info_list        *p_module_info;
} grl_module_info_pointer_list;

/**
 *  @struct Structure definition for Font Context.
 */
typedef struct svgfontcontext{
    EGLContext      eglContext;          /* Pointer to SVGContext         */
                     /* Pointer to the function pointer table of font plugin */
    struct          _grl_fontmodule_fp        *p_grl_fp_table;
    grl_module_info *p_module_info;     /* Used FontModule with informations */
              /* Pointer to internal structures depending on rendering modul */
    void            *p_fontRendererContext;
    SVGOriginType   base_line;              /* Base line type                */
    SVGEncoding     encoding;               /* Encoding                      */
    SVGFloat        angle;                  /* Angle of character            */
    SVGFloat        letter_distance;        /* Letter distance               */
    SVGFloat        font_line_space;        /* Font space                    */
    SVGUint32       settings;               /* Other setting parameter       */
    SVGError        error;        /* stores the occurance of the first error */
    SVGUint32       charMapIndex; /* index of the currently selected charmap */
    SVGBoolean      validFont;                    /* is a valid font loaded? */
    SVGFloat        opacity;      /* for app. spec. font rendering extensions */
    SVGFloat        scale_x;                /* Size scaling factor for x     */
    SVGFloat        scale_y;                /* Size scaling factor for y     */
    SVGUint32       ContourColor;            /* the outline color */
    SVGUint32       BodyColor;               /* the body color */
    SVGUint32       ContourWidth;             /* the width for contour stroker */
    SVGBoolean      ColorSet;                 /* has BodyColor been set by API? */
    SVGBoolean      WidthSet;                 /* has the ContourWidth been set by API? */  
    SVGUint8        replacement_char[8];       /* the replacement char for unknown chars */
    SVGUint32       replacement_length;
    SVGBoolean      new_replacement_char;
    SVGUint32       shares_count;           /* how many other contexts use me as shared*/
    struct svgfontcontext* shared;
    SVGInt32        thread_count;
} SVGFontContext;


/**
 *  @struct Structure definition for Font Info.
 */
typedef struct {
    SVGFloat       ascender;
    SVGFloat       descender;
    SVGFloat       underline_position;
    SVGFloat       underline_thickness;
    SVGEncoding    encoding;
    SVGFloat       font_size;
    SVGFont*       font;
    SVGUint32	   wallpaper_width_height;	
} SVGFontInfo;

typedef struct {
    SVGFloat       left;
    SVGFloat       top;
    SVGFloat       width;
    SVGFloat       height;
} SVGBBox;


/**
 *  @struct Structure definition of function pointer table of plugin modules.
 */
typedef struct _grl_fontmodule_fp{
    /* all functions */
    SVGFontContext* (*CreateFontContext) (EGLContext  drawContext, SVGFontContext* shareContext);
    SVGError (*GetFontError) (SVGFontContext*);
    SVGBoolean (*DestroyFontContext)(SVGFontContext*);
    void (*SetFontSize)(SVGFontContext*, SVGUint32);
    void (*LoadFont)(SVGFontContext*, SVGFont*, SVGEncoding, SVGUint32);
    void (*EnableFontSettings)(SVGFontContext*, SVGUint32);
    void (*DisableFontSettings)(SVGFontContext*, SVGUint32);
    SVGBoolean (*IsFontSettingEnabled)(SVGFontContext*, SVGUint32);
    void (*GetFontBitmap)(SVGFontContext*, SVGUint8*, SVGUint32, SVGUint32*, SVGFloat*, SVGFloat*);
    void (*FreeFontBitmapMem)(SVGFontContext*, SVGUint32);
    void (*DrawFontWithCursorInfo)
        (SVGFontContext*, SVGFloat, SVGFloat, SVGUint8*, SVGUint32, SVGPoint*);
    void (*GetFontBBox)
        (SVGFontContext*, SVGUint8*, SVGUint32, SVGBBox*);
    void (*get_font_suffix)(S8 **suffix_list, U32 *list_size);
    void (*fill_font_structs)(SVGUint32 *font_filenames, SVGUint32 num_files, SVGFont **fonts, SVGUint32 *max_fonts);
    SVGUint32 (*GetFontMaxChars)
        (SVGFontContext*, SVGFloat, SVGFloat, SVGUint8*, SVGUint32);
    void (*GetFontInformation)(SVGFontContext *ctx, SVGFontInfo *info);
    void (*DrawFontStringExt)
       (SVGFontContext* ctx, SVGPoint* position, SVGFloat* angle, SVGUint8* glyphs,
        SVGUint32 glyphLength);
    SVGUint32 (*GetFontBitmapExt)(SVGFontContext*, SVGUint8**, SVGUint32*, SVGUint32, SVGUint32*, SVGUint32*, SVGBBox**, SVGBBox**, SVGBBox**, SVGBBox**, SVGUint32*);
	void (*DumpFontWallpaper)(SVGFontContext* ctx, SVGUint32 settings, SVGInt8 *fileName);
	void (*SetCachableGlyphCount)(SVGFontContext* ctx, SVGUint32 glyph_count, SVGFontInfo *info);
} grl_fontmodule_fp;



/**
 *  @brief  Font Manager function prototype.
 */

/* [BEGIN SYSCALLS] */

/**
 * @fn svgInitFonts
 *
 * This function initialized the fonts.
 *
 * @param  none
 *
 * @return none
 */
void svgInitFonts( void );

/**
 * @fn svgCloseFonts
 *
 * Closes the font library.
 *
 * @param  none
 *
 * @return none
 */
   void svgCloseFonts( void );

/**
 * @fn svgUpdateFonts
 *
 * Updates the font library.
 *
 * @param  none
 *
 * @return none
 */
 void svgUpdateFonts(void);

/**
 * @fn svgGetAvailableFontModules
 *
 * Interface to get available fonts.
 *
 * @param  SVGFontModulet       : pointer to the font plugin
 * @param  SVGUint32            : module id of the font
 * @param  SVGUint8             : search control
 *
 * @return TRUE or FALSE
 */
 SVGBoolean  svgGetAvailableFontModules( SVGFontModule *font_module,
                                                SVGUint32 module_id,
                                                SVGUint32 list_control );

/**
 * @fn svgCreateFontContext
 *
 * Creates a new font context for the given font plugin module.
 *
 * @param  SVGFontModulet       : pointer to the font plugin
 * @param  drawContext          : EGL context
 * @param  shareContext         : Font context for sharing font plugin memory
 *
 * @return SVGFontContext       : the newly created font context
 */
 SVGFontContext* svgCreateFontContext( const SVGFontModule* fontPlugin,
                                              EGLContext  drawContext,
                                              SVGFontContext * shareContext);


/**
 * @fn svgDestroyFontContext
 *
 * Closes the given font context.
 *
 * @param  SVGFontContext       : Context of Font
 *
 * @return none
 */
 void    svgDestroyFontContext( SVGFontContext *ctx );



/**
 * @fn svgGetUsedFontModule
 *
 * Returns the used font module of the context.
 *
 * @param  SVGFontContext       : Context of Font
 * @param  SVGFontModule        : empty font module
 *
 * @return none
 */
 void    svgGetUsedFontModule( SVGFontContext *ctx,
                                      SVGFontModule *font_module );


/**
 * @fn svgGetFontError
 *
 * Returns the error state of the given font context.
 *
 * @param  SVGFontContext       : Context of Font
 *
 * @return SVGError             : error state of the context
 */
 SVGError   svgGetFontError( SVGFontContext *p_ctx );





/**
 * @fn svgGetAvailableFonts
 *
 * <brief description>.
 * <detailed description>
 *
 * @param  SVGFontContext       : Pointer to Font context
 * @param  SVGFont              : Pointer to font element
 * @param  SVGUint8             : The font name
 * @param  SVGUint32            : Setting of searching font
 * @param  SVGUint8             : List Control
 *
 * @return TRUE or FALSE
 */
 SVGBoolean  svgGetAvailableFonts( SVGFontContext *ctx,
                                          SVGFont *font,
                                          const SVGUint8 *fontName,
                                          SVGUint32 settings,
                                          SVGUint32 list_control );

/**
 * @fn svgLoadFont
 *
 * <brief description>.
 * <detailed description>
 *
 * @param  SVGFontContext : Pointer to Font context
 * @param  SVGFont        : Pointer to a font structure
 * @param  SVGEncoding    : The char mapping which should be used for font file
 * @param  SVGUint8       : The selected size of font
 *
 * @return none
 */
 void    svgLoadFont( SVGFontContext* ctx,
                             SVGFont* font,
                             SVGEncoding charMap,
                             SVGUint32 fontSize );

/**
 * @fn svgSetFontRotateI
 *
 * <brief description>.
 * <detailed description>
 *
 * @param  SVGFontContext       : Pointer to Font context
 * @param  SVGUint16            : The rotate angle -360<angle<360 degrees
 *
 * @return none
 */
 void    svgSetFontRotate( SVGFontContext* ctx,
                                   SVGFloat angle );

/**
 * @fn svgGetFontRotateI
 *
 * <brief description>.
 * <detailed description>
 *
 * @param  SVGFontContext       : Pointer to Font context
 *
 * @return Rotate angle
 */
 SVGFloat    svgGetFontRotate( const SVGFontContext* ctx );


/**
 * @fn svgEnableFontSettings
 *
 * <brief description>.
 * <detailed description>
 *
 * @param  SVGFontContext       : Pointer to Font context
 * @param  SVGUint32            : A combination of defined settings
 *
 * @return none
 */
 void    svgEnableFontSettings( SVGFontContext* ctx,
                                       SVGUint32 settings );

/**
 * @fn svgDisableFontSettings
 *
 * <brief description>.
 * <detailed description>
 *
 * @param  SVGFontContext       : Pointer to Font context
 * @param  SVGUint32            : A combination of defined settings
 *
 * @return none
 */
 void    svgDisableFontSettings( SVGFontContext* ctx,
                                        SVGUint32 settings );

/**
 * @fn svgIsFontSettingEnabled
 *
 * <brief description>.
 * <detailed description>
 *
 * @param  SVGFontContext       : Pointer to Font context
 * @param  SVGUint32            : A combination of defined settings
 *
 * @return TRUE or FALSE
 */
 SVGBoolean  svgIsFontSettingEnabled( SVGFontContext* ctx,
                                             SVGUint32 setting );

/**
 * @fn svgSetFontLetterDistanceI
 *
 * <brief description>.
 * <detailed description>
 *
 * @param  SVGFontContext       : Pointer to Font context
 * @param  SVGUint32            : The space between two letters in pixel
 *
 * @return none
 */
 void    svgSetFontLetterDistance( SVGFontContext* ctx,
                                           SVGFloat letterDistance );

/**
 * @fn svgGetFontLetterDistanceI
 *
 * <brief description>.
 * <detailed description>
 *
 * @param  SVGFontContext*      : Pointer to Font context
 *
 * @return The space between two letters in pixel
 */
 SVGFloat   svgGetFontLetterDistance( const SVGFontContext* ctx );

/**
 * @fn svgGetFontBBoxI
 *
 * <brief description>.
 * <detailed description>
 *
 * @param  SVGFontContext       : Pointer to Font context
 * @param  SVGUint8*            : The text to compute the bounding box
 * @param  SVGUint16            : Length of passed text
 * @param  SVGUint16*           : size X
 * @param  SVGUint16*           : size Y
 *
 * @return none
 */
 void svgGetFontBBox( SVGFontContext* ctx,
                              SVGUint8* text,
                              SVGUint32 textLength,
                              SVGBBox* textBBox );

/**
 * @fn svgGetFontLineSpacing
 *
 * <brief description>.
 * <detailed description>
 *
 * @param  SVGFontContext       : Pointer to Font context
 *
 * @return The space between two letters in pixel
 */
 SVGFloat   svgGetFontLineSpacing( const SVGFontContext* ctx );

/**
 * @fn svgSetFontDrawingOrigin
 *
 * <brief description>.
 * <detailed description>
 *
 * @param  SVGFontContext       : Pointer to Font context
 * @param  SVGOriginType        : ORIGIN_TOP/ORIGIN_BASELINE/ORIGIN_BOTTOM
 *
 * @return none
 */
 void svgSetFontDrawingOrigin( SVGFontContext* ctx,
                                      SVGOriginType originType );


/**
 * @fn svgGetFontDrawingOrigin
 *
 * <brief description>.
 * <detailed description>
 *
 * @param  SVGFontContext       : Pointer to Font context
 *
 * @return The origin type (ORIGIN_TOP/ORIGIN_BASELINE/ORIGIN_BOTTOM)
 */
 SVGOriginType    svgGetFontDrawingOrigin( const SVGFontContext* ctx );


/**
 * @fn svgGetFontBitmap
 *
 * <brief description>.
 * <detailed description>
 *
 * @param  SVGFontContext*      : Pointer to Font context
 * @param  SVGUint8*            : Pointer to String of passed text
 * @param  SVGUint16            : Length of passed text
 * @param  SVGSurface**         : Surface which contains the font alpha mask
 *
 * @return none
 */
 void svgGetFontBitmap( SVGFontContext   *ctx,
                               SVGUint8         *text,
                               SVGUint32        textLength,
                               SVGUint32		*texture,
                               SVGFloat         *width,
                               SVGFloat         *height);

/**
 * @fn svgFreeFontBitmapMem
 *
 * <brief description>.
 * <detailed description>
 *
 * @param  SVGFontContext       : Pointer to Font context
 * @param  SVGSurface           : Pointer to surface
 *
 * @return none
 */
 void svgFreeFontBitmapMem( SVGFontContext *ctx,
                                   SVGUint32      texture );

/**
 * @fn svgDrawFontStringI
 *
 * <brief description>.
 * <detailed description>
 *
 * @param  SVGFontContext*      : Pointer to Font context
 * @param  SVGInt16             : Drawing position X
 * @param  SVGInt16             : Drawing position Y
 * @param  SVGUint8*            : Pointer to text
 * @param  SVGUint16            : Length of passed text
 *
 * @return none
 */
 void svgDrawFontString( SVGFontContext* ctx,
                                 SVGFloat positionX,
                                 SVGFloat positionY,
                                 SVGUint8* text,
                                 SVGUint32 textLength );

/**
 * @fn svgDrawFontWithCursorInfoI
 *
 * <brief description>.
 * <detailed description>
 *
 * @param  SVGFontContext*      : Pointer to Font context
 * @param  SVGInt16             : Drawing position X
 * @param  SVGInt16             : Drawing position Y
 * @param  SVGUint8*            : Pointer to text
 * @param  SVGUint16            : Length of passed text
 *
 * @return none
 */
 void svgDrawFontWithCursorInfo( SVGFontContext* ctx,
                                        SVGFloat positionX,
                                        SVGFloat positionY,
                                        SVGUint8* text,
                                        SVGUint32 textLength,
                                        SVGPoint* cursorPosArray);

/**
 * @fn svgGetFontMaxChars
 *
 * <brief description>.
 * <detailed description>
 *
 * @param  SVGFontContext*  ctx         : Pointer to Font context
 * @param  SVGInt16         maxspace    : maximum space for text drawing
 * @param  SVGInt16*        text        : text to be drawn
 * @param  SVGUint16        text_length : number of chars in text
 *
 * @return SVGUint16                    : number of chars matching
 *                                        into the given size
 */
 SVGUint32   svgGetFontMaxChars( SVGFontContext  *ctx,
                                        SVGFloat        max_space_x,
                                        SVGFloat        max_space_y,
                                        SVGUint8        *text,
                                        SVGUint32       textLength);


/**
 * @fn svgGetFontInformation
 *
 * This function returns the ascent and descent of the selected font/size.
 *
 * @param  SVGFontContext*  ctx         : Pointer to Font context
 * @param  SVGInt16*        ascent      : ascent (from baseline to top)
 * @param  SVGInt16*        descent     : descent (from baseline to bottom)
 *
 * @return none
 */
void svgGetFontInformation ( SVGFontContext *ctx, SVGFontInfo *info );

/**
 * @fn svgSetFontOpacity
 *
 * Sets the opacity of the drawn font.
 *
 * @param  SVGFontContext       : Pointer to Font context
 * @param  SVGUint8             : Opacity value to be multiplied with per-pixel opacity
 *          of the font bitmap. A value of 255 means no transparency (normal font drawing).
 *
 * @return none
 */
 void    svgSetFontOpacity( SVGFontContext* ctx,
                                   SVGFloat		opacity);

/**
 * @fn svgGetFontOpacity
 *
 * Gets the opacity of the drawn font.
 *
 * @param  SVGFontContext       : Pointer to Font context
 *
 * @return Opacity value
 */
 SVGFloat    svgGetFontOpacity( SVGFontContext* ctx);

/**
 * @fn svgSetFontSize
 *
 * Sets new size of drawn font
 *
 * @param  SVGFontContext       : Pointer to Font context
 * @param  SVGUint8             : new font size
 *
 * @return none
 */
void svgSetFontSize( SVGFontContext *ctx, SVGUint32 fontSize );

/**
 * @fn svgDrawFontStringI
 *
 * <brief description>.
 * <detailed description>
 *
 * @param  SVGFontContext*      : Pointer to Font context
 * @param  SVGPoint             : Drawing position
 * @param  SVGInt16             : Angle of the glyph
 * @param  SVGUint8*            : Pointer to text
 * @param  SVGUint16            : Length of passed text
 * @param  SVGBoolean           : Use CPU or GPU for drawing
 *
 * @return none
 */
void svgDrawFontStringExt( SVGFontContext* ctx,
                               SVGPoint* position,
                               SVGFloat* angle,
                               SVGUint8* glyphs,
                               SVGUint32 glyphLength);


/**
 * \fn svgSetFontScale
 *
 * Set a scaling factor for the font sizes in the x and y dimensions.
 *
 * \param   SVGFontContext     *ctx     : Pointer to Font context
 * \param   SVGFloat           scale_x  : Scaling factor for x
 * \param   SVGFloat           scale_y  : Scaling factor for y
 *
 * \return none
 */
void svgSetFontScale( SVGFontContext *ctx,
                             SVGFloat       scale_x,
                             SVGFloat       scale_y );


/**
 * \fn svgGetFontScale
 *
 * Returns the current scaling factors for the font sizes in the x and y dimensions.
 *
 * \param   SVGFontContext     *ctx     : Pointer to Font context
 * \param   SVGFloat           *scale_x : Pointer for returning scaling factor for x
 * \param   SVGFloat           *scale_y : Pointer for returning scaling factor for y
 *
 * \return none
 */
void svgGetFontScale( SVGFontContext *ctx,
                             SVGFloat       *scale_x,
                             SVGFloat       *scale_y );

/**
 * \fn svgSetContourColor
 *
 * Set the replacement char for unknown chars in string(depending on Encoding setting)
 *
 * \param   SVGFontContext     *ctx     : Pointer to Font context
 * \param   SVGUint32          color    : color for outline and 2color strings
 *
 * \return none
 */
void svgSetContourColor (SVGFontContext *ctx, SVGUint32     color);

/**
 * \fn svgGetContourColor
 *
 * Set the replacement char for unknown chars in string(depending on Encoding setting)
 *
 * \param   SVGFontContext     *ctx     : Pointer to Font context
 *
 * \return SVGUint32: The color for outline and 2color strings
 */
SVGUint32 svgGetContourColor (SVGFontContext    *ctx);

/**
 * \fn svgSetContourColor
 *
 * Set the replacement char for unknown chars in string(depending on Encoding setting)
 *
 * \param   SVGFontContext     *ctx     : Pointer to Font context
 * \param   SVGUint32          color    : color for outline and 2color strings
 *
 * \return none
 */
void svgSetBodyColor (SVGFontContext *ctx, SVGUint32     color);

/**
 * \fn svgGetContourColor
 *
 * Set the replacement char for unknown chars in string(depending on Encoding setting)
 *
 * \param   SVGFontContext     *ctx     : Pointer to Font context
 *
 * \return SVGUint32: The color for outline and 2color strings
 */
SVGUint32 svgGetBodyColor (SVGFontContext    *ctx);

/**
 * \fn svgSetContourColor
 *
 * Set the replacement char for unknown chars in string(depending on Encoding setting)
 *
 * \param   SVGFontContext     *ctx     : Pointer to Font context
 * \param   SVGUint32          color    : color for outline and 2color strings
 *
 * \return none
 */
void svgSetContourWidth (SVGFontContext *ctx, SVGUint32     width);

/**
 * \fn svgGetContourColor
 *
 * Set the replacement char for unknown chars in string(depending on Encoding setting)
 *
 * \param   SVGFontContext     *ctx     : Pointer to Font context
 *
 * \return SVGUint32: The color for outline and 2color strings
 */
SVGUint32 svgGetContourWidth (SVGFontContext    *ctx);

/**
 * \fn svgSetFontReplacementChar
 *
 * Set the replacement char for unknown chars in string(depending on Encoding setting)
 *
 * \param   SVGFontContext     *ctx     : Pointer to Font context
 * \param   SVGUint8*          replace  : Pointer to ReplacementChar
 * \param   SVGUint32          length   : length of char array
 *
 * \return none
 */
void svgSetFontReplacementChar( SVGFontContext *ctx,
                                       SVGUint8*      replace,
                                       SVGUint32      length );


/**
 * \fn svgGetFontReplacementChar
 *
 * Returns the replacement char for unknown chars in string(depending on Encoding setting)
 *
 * \param   SVGFontContext     *ctx     : Pointer to Font context
 * \param   SVGFloat           *scale_x : Pointer for returning scaling factor for x
 * \param   SVGFloat           *scale_y : Pointer for returning scaling factor for y
 *
 * \return none
 */
void svgGetFontReplacementChar( SVGFontContext *ctx,
                                       SVGUint8*      replace,
                                       SVGUint32      *length );


/**
 * @fn svgGetFontBitmapExt
 *
 * <brief description>.
 * <detailed description>
 *
 * @param  SVGFontContext*      : Pointer to Font context
 * @param  SVGUint8*            : Pointer to String of passed text
 * @param  SVGUint              : Length of passed text
 * @param  SVGUint32*           : pointer to returned texture for filled glyphs
 * @param  SVGUint32*           : pointer to returned texture for stroked glyphs
 * @param  SVGBBox*             : pointer-array of returned texture coordinates of filled glyphs
 * @param  SVGBBox*             : pointer-array of returned texture coordinates of stroked glyphs
 * @param  SVGBBox*             : pointer-array of returned texture coordinates of filled glyphs
 * @param  SVGBBox*             : pointer-array of returned texture coordinates of stroked glyphs
 *
 * @return number of glyphs to be drawn(equal or less then textLength
 */
SVGUint32 svgGetFontBitmapExt( SVGFontContext   *ctx,
                                      SVGUint8         **text,
                                      SVGUint32        *textLength,
                                      SVGUint32	   	   textNumber,
                                      SVGUint32*       filledTexture,
									  SVGUint32*       strokedTexture,											
                                      SVGBBox**        filledTextureBox,
                                      SVGBBox**        strokedTextureBox,
                                      SVGBBox**        filledGlyphBox,
                                      SVGBBox* *       strokedGlyphBox,
                                      SVGUint32*       glyphNumber);
	
/**
 * @fn svgDumpFontWallpaper
 *
 * <brief description>.
 * <detailed description>
 *
 * @param  SVGFontContext*     ctx: Pointer to Font context
 * @param  SVGUint32      settings: Sets which wallpaper should be dumped(
 *									SVG_FONT_OUTLINE for only outline wallpaper
 *									SVG_FONT_2COLOR  for both
 *									any other value for only filled wallpaper										
 * @param  SVGInt32       fileName: Name with full path to the file 
									(_FILLED.ppm or _STROKED.ppm) will be added to the filename 
									by the api, depending on the settings.
									if fileName is null default fileName if used(FontWallpaper_*)
 * @return none
 */	
void svgDumpFontWallpaper(  SVGFontContext* ctx, 
							SVGUint32 		settings, 
							SVGInt8			*fileName);

/**
 * @fn svgSetCachableGlyphCount
 *
 * <brief description>.
 * <detailed description>
 *
 * @param  SVGFontContext* ctx:         Pointer to Font context
 * @param  SVGUint32       glyph_count: number of glyphs which have to fit into the
 *                                      "cache" texture will lead to recalculation of
 *                                      texture size(when bigger than before)
 *                                      When cache is in use it will be cleaned completely
 *                                      on growing!
 * @param  SVGFontInfo*    info:        can be NULL, when not NULL contains
 *                                      e.g. info about newly allocated
 *                                      texture size
 * @return none
 */
void svgSetCacheableGlyphCount(SVGFontContext *ctx,
							  SVGUint32      glyph_count,
							  SVGFontInfo    *info);
							  

/**
 * @fn svgGetFontGlyphCount
 *
 * <brief description>.
 * <detailed description>
 *
 * @param  SVGFontContext* ctx:        Pointer to Font context
 * @param  SVGUint8         *text: text to be rendered 
 * @param  SVGUint32        textLength: text buffer length
 * @return none
 */

SVGUint32 svgGetFontGlyphCount ( SVGFontContext   *ctx,
	                             SVGUint8         *text,
	                             SVGUint32        textLength);

/**
 * @fn svgEnableShapingEng
 *
 * <brief description>.
 * <detailed description>
 *
 * @param  SVGLangScript         langscript: langscript to be enabled
 * @param  SVGBoolean        autolang_detect_enable: enable globally auto lang detection process
 * @return SVG_TRUE if success else SVG_FALSE
 */

SVGBoolean svgEnableShapingEng(SVGLangScript langscript,
                               SVGBoolean    autolang_detect_enable);

/**
 * @fn svgEnableShapingEng
 *
 * <brief description>.
 * <detailed description>
 *
 * @param  SVGLangScript         langscript: langscript to be disabled
 * @param  SVGBoolean        autolang_detect_disable: disable globally auto lang detection process
 * @return SVG_TRUE if success else SVG_FALSE
 */

SVGBoolean svgDisableShapingEng(SVGLangScript langscript,
                               SVGBoolean    autolang_detect_disable);


#ifdef __cplusplus
}
#endif
#endif    /* SVG_FONT_H */

